#include <WiFi.h>
#include <AccelStepper.h>

// ================= WIFI =================
const char* ssid     = "NOMBRE RED WIFI";
const char* password = "CLAVE WIFI ";

IPAddress local_IP(192,168,0,50);
IPAddress gateway(192,168,0,1);
IPAddress subnet(255,255,255,0);

// Puerto GS-232A
WiFiServer server(4533);
WiFiClient client;

// ================= CNC SHIELD ===========
#define X_STEP_GPIO 26
#define X_DIR_GPIO  16
#define Y_STEP_GPIO 25
#define Y_DIR_GPIO  17
#define EN_GPIO     27

AccelStepper azMotor(AccelStepper::DRIVER, X_STEP_GPIO, X_DIR_GPIO);
AccelStepper elMotor(AccelStepper::DRIVER, Y_STEP_GPIO, Y_DIR_GPIO);

// Calibración
const float stepsPerDegreeAZ = 18.0;
const float stepsPerDegreeEL = 18.0;

const int maxSpeed = 3000;
const int accel    = 1500;

String cmdUSB = "";
String cmdTCP = "";

enum InChannel { CH_USB, CH_TCP };
InChannel replyChannel = CH_USB;

// ---------------------------------------------------------
void setup() {
  Serial.begin(115200);
  delay(200);

  pinMode(EN_GPIO, OUTPUT);
  digitalWrite(EN_GPIO, LOW);

  azMotor.setMaxSpeed(maxSpeed);
  azMotor.setAcceleration(accel);
  elMotor.setMaxSpeed(maxSpeed);
  elMotor.setAcceleration(accel);

  WiFi.mode(WIFI_STA);
  WiFi.config(local_IP, gateway, subnet);
  WiFi.begin(ssid, password);

  Serial.print("Conectando WiFi");
  while (WiFi.status() != WL_CONNECTED) {
    delay(300);
    Serial.print(".");
  }
  Serial.println();
  Serial.print("IP: ");
  Serial.println(WiFi.localIP());

  server.begin();
  server.setNoDelay(true);
  Serial.println("GS-232A TCP LISTO");
}

// ---------------------------------------------------------
void loop() {
  // --- SI EL CLIENTE EXISTE PERO SE CAYÓ → CERRARLO ---
  if (client && !client.connected()) {
    client.stop();
  }

  // --- ACEPTAR NUEVO CLIENTE SIEMPRE ---
  WiFiClient newClient = server.available();
  if (newClient) {
    client.stop();      // CIERRE FORZADO
    client = newClient;
    cmdTCP = "";
    Serial.println("CLIENTE TCP NUEVO");
  }

  processUSB();
  processTCP();

  azMotor.run();
  elMotor.run();
}

// ---------------------------------------------------------
void sendOut(const char* s) {
  if (replyChannel == CH_USB) {
    Serial.print(s);
  } else {
    if (client && client.connected()) client.print(s);
  }
}

// ---------------------------------------------------------
void sendPosition() {
  int az = (int)lround(azMotor.currentPosition() / stepsPerDegreeAZ);
  int el = (int)lround(elMotor.currentPosition() / stepsPerDegreeEL);

  az = constrain(az, 0, 450);
  el = constrain(el, 0, 180);

  char buffer[24];
  sprintf(buffer, "+0%03d+0%03d\r\n", az, el);
  sendOut(buffer);
}

// ---------------------------------------------------------
void hardStop() {
  azMotor.stop();
  elMotor.stop();
  azMotor.moveTo(azMotor.currentPosition());
  elMotor.moveTo(elMotor.currentPosition());
}

// ---------------------------------------------------------
void handleCommand(String s) {
  s.trim();
  s.toUpperCase();

  if (s == "C" || s == "C2") { sendPosition(); return; }
  if (s == "S") { hardStop(); sendPosition(); return; }

  if (s.startsWith("W")) {
    int i = s.indexOf(' ');
    if (i > 0) {
      int azDeg = s.substring(1, i).toInt();
      int elDeg = s.substring(i + 1).toInt();

      azMotor.moveTo(lround(azDeg * stepsPerDegreeAZ));
      elMotor.moveTo(lround(elDeg * stepsPerDegreeEL));

      sendPosition();
      return;
    }
  }

  sendOut("?\r\n");
}

// ---------------------------------------------------------
void processUSB() {
  while (Serial.available()) {
    char c = Serial.read();
    if (c == '\n' || c == '\r') {
      if (cmdUSB.length()) {
        replyChannel = CH_USB;
        handleCommand(cmdUSB);
        cmdUSB = "";
      }
    } else cmdUSB += c;
  }
}

// ---------------------------------------------------------
void processTCP() {
  if (!(client && client.connected())) return;

  while (client.available()) {
    char c = client.read();
    if (c == '\n' || c == '\r') {
      if (cmdTCP.length()) {
        replyChannel = CH_TCP;
        handleCommand(cmdTCP);
        cmdTCP = "";
      }
    } else cmdTCP += c;
  }
}
